header {* Type classes in Isabelle *}

theory Type_Classes
imports Main
begin

text {*
  \<bullet> Isabelle type-classes are based on the original version of
    Haskell98, which is conceptionally very simple.

  \<bullet> The tight integration with locales and the general local theory
    infrastructure provides some extra flexibility.

  \<bullet> Our running example is about basic algebraic structures
    (semigroups, monoids, groups).

  \<bullet> Subsequently we show types and sort constraints explicitly.
*}

declare [[show_types, show_sorts]]


section {* Class definition and abstract reasoning *}

class semigroup =
  fixes mult :: "'a \<Rightarrow> 'a \<Rightarrow> 'a"  (infixl "\<odot>" 70)
  assumes assoc: "(x \<odot> y) \<odot> z = x \<odot> (y \<odot> z)"

text {*
  This defines a predicate together with rich infrastructure for
  definitions and proofs "in" a local context.
*}

term semigroup
thm semigroup_def

thm assoc

definition (in semigroup)
  "square x = x \<odot> x"

lemma (in semigroup)
  square_square: "square (square x) = x \<odot> x \<odot> x \<odot> x"
  unfolding square_def by (simp add: assoc)

thm square_square


text {* Alternative "context" specification: *}

context semigroup
begin

definition "cube x = x \<odot> x \<odot> x"

lemma cube: "cube x = square x \<odot> x"
  unfolding square_def cube_def by simp

end


section {* Concrete instantiations *}

text {*
  Instances for particular type constructions require to
  specify the "fixes" and prove the "assumes" part.
*}

instantiation nat :: semigroup
begin

print_context

fun mult_nat :: "nat \<Rightarrow> nat \<Rightarrow> nat"
where
  "(0::nat) \<odot> n = n"
| "Suc m \<odot> n = Suc (m \<odot> n)"

instance
proof
  fix a b c :: nat 
  show "(a \<odot> b) \<odot> c = a \<odot> (b \<odot> c)"
    by (induct a) auto
qed

end

text {*
  Now we can re-use derived specifications and proofs from @{class semigroup}
  for type @{typ nat}.
*}

thm assoc [where 'a = nat]
thm square_square [where 'a = nat]
thm cube [where 'a = nat]

text {*
  \<bullet> The system produces such type-instances implicitly via unification.

  \<bullet> Unexpected failure to apply some rule is often caused by
    too general types in the statements.
*}


subsection {* Exercise *}

text {*
  \<triangleright> Instantiate @{class semigroup} for @{typ int}, @{typ "'a list"}
    with suitable definitions of @{term mult}.
*}


section {* Subclasses *}

text {*
  Subclasses can be defined by importing existing classes.
  Multiple imports are also possible.
*}

class monoidl = semigroup +
  fixes neutral :: 'a  ("\<one>")
  assumes neutl: "\<one> \<odot> x = x"

class monoid = monoidl +
  assumes neutr: "x \<odot> \<one> = x"

text {*
  Instantiation works as before, while previous super-class instances
  are taken into account.  Only the remaining definitions and proofs
  need to be given.
*}

instantiation nat :: monoidl
begin

print_context

definition neutral_nat_def: "\<one> = (0::nat)"

instance
proof
  fix a :: nat
  show "\<one> \<odot> a = a" unfolding neutral_nat_def by simp
qed

end

instantiation nat :: monoid 
begin

instance
proof
  fix a :: nat
  show "a \<odot> \<one> = a"
    unfolding neutral_nat_def by (induct a) simp_all
qed

end


section {* Abstract instantiations *}

text {*
  \<bullet> Extending the class hierarchy naturally leads to derivable "edges".

  \<bullet> Subclass relations can be proven later on, and are thus registered
    to the local theory infrastructure, type-inference etc.
*}

class group = monoidl +
  fixes inverse :: "'a \<Rightarrow> 'a"  ("_\<div>" [999] 1000)
  assumes invl: "x\<div> \<odot> x = \<one>"

lemma (in group) left_cancel: "x \<odot> y = x \<odot> z \<longleftrightarrow> y = z"
proof
  assume "x \<odot> y = x \<odot> z"
  then have "x\<div> \<odot> (x \<odot> y) = x\<div> \<odot> (x \<odot> z)" by simp
  then have "(x\<div> \<odot> x) \<odot> y = (x\<div> \<odot> x) \<odot> z" by (simp add: assoc)
  then show "y = z" by (simp add: neutl invl)
next
  assume "y = z"
  then show "x \<odot> y = x \<odot> z" by simp
qed

subclass (in group) monoid
proof
  fix x
  have "x\<div> \<odot> x = \<one>" by (rule invl)
  then have "x\<div> \<odot> (x \<odot> \<one>) = x\<div> \<odot> x"
    by (simp add: assoc [symmetric] neutl invl)
  then show "x \<odot> \<one> = x" by (simp add: left_cancel)
qed


section {* Further reading *}

text {*
  See http://isabelle.in.tum.de/dist/Isabelle/doc/classes.pdf
  and http://isabelle.in.tum.de/dist/Isabelle/doc/locales.pdf

  The class infrastructure is an extension of that for locales,
  with additional support for type-inference.
*}

end

