header {* Structured Natural Deduction in Isar *}

theory Natural_Deduction
imports Main
begin

section {* Rule statements *}

text {*
  Isabelle/Pure "theorems" are always natural deduction rules,
  which sometimes happen to consist of a conclusion only.

  \<And> and \<Longrightarrow> indicate the rule structure.
*}

thm conjI
thm impI
thm nat.induct

text {*
  The object-logic is embedded into the Pure framework via an implicit
  derivability judgment @{term "Trueprop :: bool \<Rightarrow> prop"}.

  Thus any HOL formulae appears atomic to the Pure framework, while
  the rule structure outlines the corresponding proof pattern.

  This can be made explicit as follows:
*}

notepad
begin
  write Trueprop  ("Tr")

  thm conjI
  thm impI
  thm nat.induct
end

text {*
  Isar provides first-class notation for rule statements as follows.
*}

print_statement conjI
print_statement impI
print_statement nat.induct


subsection {* Exercise *}

text {*
  \<triangleright> Write Pure and Isar rule statements of introductions and eliminations
    for some connectives, e.g.

      @{prop "\<not> P"}
      @{prop "P \<and> Q"}
      @{prop "P \<or> Q"}
      @{prop "\<forall>x. P x"}
      @{prop "\<exists>x. P x"}

      @{prop "x \<in> A \<inter> B"}
      @{prop "x \<in> A \<union> B"}

  Hint: The rules can be proven "by blast" as a sanity check.
*}


section {* Isar context elements *}

text {*
  Main idea: use vacuous Isar proof body as block-structured
  formal notepad and derive some results out of the blue.
*}

notepad
begin
  {
    fix x
    have "B x" sorry
  }
  note `\<And>x. B x`

next

  {
    assume A
    have B sorry
  }
  note `A \<Longrightarrow> B`

next

  {
    def x \<equiv> t
    have "B x" sorry
  }
  note `B t`

next

  {
    obtain x :: 'a where "B x" sorry
    have C sorry
  }
  note `C`

end


subsection {* Exercise *}

text {*
  \<triangleright> Reproduce some of the rules from the last excercise within a
    proof body as follows.
*}

notepad
begin
  {
    fix P and Q
    assume P and Q
    then have "P \<and> Q" by blast
  }
  have "\<And>P Q. P \<Longrightarrow> Q \<Longrightarrow> P \<and> Q" by fact
end


section {* Pure rule composition *}

text {*
  The Pure framework provides means for:

    \<bullet> backward-chaining of rules by "resolution"

    \<bullet> closing of branches by "assumption"

  Both principles involve higher-order unification of
  \<lambda>-terms modulo \<alpha>\<beta>\<eta> (cf. Huet and Miller).
*}

notepad
begin
  assume a: A and b: B
  thm conjI
  thm conjI [of A B]  -- "instantiation"
  thm conjI [of A B, OF a b]  -- "instantiation and composition"
  thm conjI [OF a b]  -- "composition via unification (trivial)"
  thm conjI [OF `A` `B`]

  thm conjI [OF disjI1]
end

text {*
  Note: Low-level rule composition is tedious and leads to
  unreadable/unmaintainable expressions in the text.
*}


section {* Structured backward reasoning *}

text {*
  Idea: Canonical proof decomposition via fix/assume/show, where
  the body produces a natural deduction rule to refine some goal.
*}

notepad
begin
  fix A B :: "'a \<Rightarrow> bool"

  have "\<And>x. A x \<Longrightarrow> B x"
  proof -
    fix x
    assume "A x"
    show "B x" sorry
  qed

  have "\<And>x. A x \<Longrightarrow> B x"
  proof -
    {
      fix x
      assume "A x"
      show "B x" sorry
    } -- "implicit block structure made explicit"
    note `\<And>x. A x \<Longrightarrow> B x`
      -- "side exit for the resulting rule"
  qed
end


subsection {* Exercise *}

text {*
  \<triangleright> Produce proof decompositions for the following synthetic statement:

    @{prop "\<And>x y. A x \<Longrightarrow> B y \<Longrightarrow> C x y"}

  \<triangleright> Which variations are possible?
*}


section {* Structured rule application *}

text {*
  Idea: Previous facts and new claims are composed with a rule from
  the context (or background library).
*}

notepad
begin
  assume r1: "A \<Longrightarrow> B \<Longrightarrow> C"  -- {* simple rule (Horn clause) *}

  have A sorry  -- "prefix of facts via outer sub-proof"
  then have C
  proof (rule r1)
    show B sorry  -- "remaining rule premises via inner sub-proof"
  qed

  have C
  proof (rule r1)
    show A sorry
    show B sorry
  qed

  have A and B sorry
  then have C
  proof (rule r1)
  qed

  have A and B sorry
  then have C by (rule r1)

next

  assume r2: "A \<Longrightarrow> (\<And>x. B1 x \<Longrightarrow> B2 x) \<Longrightarrow> C"  -- {* nested rule *}

  have A sorry
  then have C
  proof (rule r2)
    fix x
    assume "B1 x"
    show "B2 x" sorry
  qed

  txt {*
    The compound rule premise @{prop "\<And>x. B1 x \<Longrightarrow> B2 x"} is better
    addressed via fix/assume/show in the body.
  *}
end


section {* Example: predicate logic *}

text {*
  Using the above principles, standard introduction and elimination proofs
  of predicate logic connectives of HOL work as follows.

  Notes:

  \<bullet> method "rule" can retrieve implicit rules declared in the library

  \<bullet> "proof" without method defaults to "proof rule"

  \<bullet> "by m1 m2" abbreviates "proof m1 qed m2"

  \<bullet> ".." abbreviates "by rule"

  \<bullet> "." abbreviates "by this"
*}


subsection {* Exercise *}

text {*
  \<triangleright> Produce some Isar proofs involving the natural deduction rules
    of \<not>, \<and>, \<or>, \<forall>, \<exists>, \<inter>, \<union> etc. seen before, using rules implicitly
    from the background library.

  \<triangleright> Which variations are possible?
*}


subsection {* Exercise *}

text {*
  \<triangleright> Give some variations of the above proof patterns.

  \<triangleright> Give analogous patterns for operators of set-theory, e.g.

    @{prop "x \<in> A \<inter> B"}
    @{prop "x \<in> A \<union> B"}
    @{prop "x \<in> \<Inter>A"}
    @{prop "x \<in> \<Union>A"}
*}

notepad
begin
  have "A \<and> B"
  proof  -- {* two strictly isolated subproofs *}
    show A sorry
  next
    show B sorry
  qed

  have "A \<and> B"
  proof  -- {* one simulatenous sub-proof *}
    show A and B sorry
  qed

  have "A \<and> B"
  proof  -- {* two subproofs in the same context *}
    show A sorry
    show B sorry
  qed

  have "A \<and> B"
  proof  -- {* swapped order *}
    show B sorry
    show A sorry
  qed

  have "A \<and> B"
  proof  -- {* sequential subproofs *}
    show A sorry
    show B using `A` sorry
  qed
end

notepad
begin
  have "x \<in> A" and "x \<in> B" sorry
  then have "x \<in> A \<inter> B" ..

  have "x \<in> A" sorry
  then have "x \<in> A \<union> B" ..

  have "x \<in> B" sorry
  then have "x \<in> A \<union> B" ..

  have "x \<in> A \<union> B" sorry
  then have C
  proof
    assume "x \<in> A"
    then show C sorry
  next
    assume "x \<in> B"
    then show C sorry
  qed

next
  have "x \<in> \<Inter>A"
  proof
    fix a
    assume "a \<in> A"
    show "x \<in> a" sorry
  qed

  have "x \<in> \<Inter>A" sorry
  then have "x \<in> a"
  proof
    show "a \<in> A" sorry
  qed

  have "a \<in> A" and "x \<in> a" sorry
  then have "x \<in> \<Union>A" ..

  have "x \<in> \<Union>A" sorry
  then obtain a where "a \<in> A" and "x \<in> a" ..
end

notepad
begin
  have "A \<and> B \<longrightarrow> B \<and> A"
  proof (rule impI)
    assume "A \<and> B"
    then show "B \<and> A"
    proof (rule conjE)
      assume a: A and b: B
      from b and a show "B \<and> A" by (rule conjI)
    qed
  qed

  have "A \<and> B \<longrightarrow> B \<and> A"
  proof rule
    assume "A \<and> B"
    then show "B \<and> A"
    proof rule
      assume a: A and b: B
      from b and a show "B \<and> A" by rule
    qed
  qed

  have "A \<and> B \<longrightarrow> B \<and> A"
  proof
    assume "A \<and> B"
    then show "B \<and> A"
    proof
      assume a: A and b: B
      from b and a show "B \<and> A" ..
    qed
  qed

  have "A \<and> B \<longrightarrow> B \<and> A"
  proof
    assume "A \<and> B"
    then obtain B and A ..
    then show "B \<and> A" ..
  qed
end

end
