header {* Structured induction proofs *}

theory Induction
imports Main
begin

section {* Induction as Natural Deduction *}

text {*
  In principle induction is just a special case of Natural Deduction.
*}

thm nat.induct
print_statement nat.induct

notepad
begin
  fix n :: nat
  have "P n"
  proof (rule nat.induct)  -- {* fragile rule application! *}
    show "P 0" sorry
  next
    fix n :: nat
    assume "P n"
    show "P (Suc n)" sorry
  qed
end

text {*
  In practice, much more proof infrastructure is required.

  The proof method "induct" provides:

  \<bullet> implicit rule selection and robust instantiation

  \<bullet> context elements via symbolic case names

  \<bullet> support for rule-structured induction statements, with
    local parameters, premises, etc.
*}

notepad
begin
  fix n :: nat
  have "P n"
  proof (induct n)
    case 0
    show ?case sorry
  next
    case (Suc n)
    from Suc.hyps show ?case sorry
  qed
end


subsection {* Exercise *}

text {*
  \<triangleright> Produce structured proofs of the following statements, using
    induction and calculational reasoning in Isar.

  Hints:

    \<bullet> Isabelle/HOL numerals are polymorphic!

    \<bullet> Terminal justifications should work "by simp".
*}

lemma fixes n :: nat shows "2 * (\<Sum>i=0..n. i) = n * (n + 1)"
  sorry

text {*
  This formulation is more conventional, with explicit
  division on the RHS.
*}

lemma fixes n :: nat shows "(\<Sum>i=0..n. i) = n * (n + 1) div 2"
  sorry

text {*
  Here we state divisibility in the result expression explicitly.
*}

lemma fixes n :: nat shows "\<exists>k. n * (n + 1) = 2 * k"
  sorry


section {* Induction with local parameters and premises *}

text {*
  Idea: Pure rule statements are passed through the induction rule.
  This achieves convenient proof patterns, thanks to some internal
  trickery in the "induct" method.

  Important: Using compact HOL formulae with \<forall>/\<longrightarrow> is a well-known
  anti-pattern! It would produce lots of useless formal noise.
*}

notepad
begin
  fix n :: nat
  fix P :: "nat \<Rightarrow> bool"
  fix Q :: "'a \<Rightarrow> nat \<Rightarrow> bool"

  have "P n"
  proof (induct n)
    case 0
    show "P 0" sorry
  next
    case (Suc n)
    from `P n` show "P (Suc n)" sorry
  qed

  have "A n \<Longrightarrow> P n"
  proof (induct n)
    case 0
    from `A 0` show "P 0" sorry
  next
    case (Suc n)
    from `A n \<Longrightarrow> P n`
      and `A (Suc n)` show "P (Suc n)" sorry
  qed

  have "\<And>x. Q x n"
  proof (induct n)
    case 0
    show "Q x 0" sorry
  next
    case (Suc n)
    from `\<And>x. Q x n` show "Q x (Suc n)" sorry
    txt {* local quantification admits arbitrary instances *}
    note `Q a n` and `Q b n`
  qed  
end


subsection {* Exercise *}

text {*
  \<triangleright> Give the combined pattern for @{prop "\<And>x. A x n \<Longrightarrow> Q x n"}
    in the proof context above.
*}


section {* Implicit induction context *}

text {*
  The "induct" method can isolate local parameters and premises
  directly from the given statement.

  This is convenient in practical applications, but requires some
  understanding of what is really going on (as explained above).
*}

notepad
begin
  fix n :: nat
  fix Q :: "'a \<Rightarrow> nat \<Rightarrow> bool"

  fix x :: 'a
  assume "A x n" then have "Q x n"
  proof (induct n arbitrary: x)
    case 0
    from `A x 0` show "Q x 0" sorry
  next
    case (Suc n)
    from `\<And>x. A x n \<Longrightarrow> Q x n`
      and `A x (Suc n)` show "Q x (Suc n)" sorry
  qed  
end


section {* Advanced induction with term definitions *}

text {*
  Induction over subexpressions of a certain shape are delicate to
  formalize.  The Isar "induct" method provides infrastructure for this.

  Idea: sub-expressions of the problem are turned into a
  defined induction variable; often accompanied with fixing
  of auxiliary parameters in the original expression.
*}

notepad
begin
  fix a :: "'a \<Rightarrow> nat"
  fix A :: "nat \<Rightarrow> bool"

  assume "A (a x)"
  then have "P (a x)"
  proof (induct "a x" arbitrary: x)
    case 0
    note prem = `A (a x)`
      and defn = `0 = a x`
    show "P (a x)" sorry
  next
    case (Suc n)
    note hyp = `\<And>x. n = a x \<Longrightarrow> A (a x) \<Longrightarrow> P (a x)`
      and prem = `A (a x)`
      and defn = `Suc n = a x`
    show "P (a x)" sorry
  qed
end


section {* Example: finite sequences *}

text {*
  Regard the following formalization of finite sequences with some
  standard operations.
*}

datatype 'a seq = Empty | Seq 'a "'a seq"

fun conc :: "'a seq \<Rightarrow> 'a seq \<Rightarrow> 'a seq"
where
  "conc Empty ys = ys"
| "conc (Seq x xs) ys = Seq x (conc xs ys)"

fun reverse :: "'a seq \<Rightarrow> 'a seq"
where
  "reverse Empty = Empty"
| "reverse (Seq x xs) = conc (reverse xs) (Seq x Empty)"

lemma conc_empty: "conc xs Empty = xs"
  by (induct xs) simp_all

lemma conc_assoc: "conc (conc xs ys) zs = conc xs (conc ys zs)"
  by (induct xs) simp_all

lemma reverse_conc: "reverse (conc xs ys) = conc (reverse ys) (reverse xs)"
  by (induct xs) (simp_all add: conc_empty conc_assoc)

lemma reverse_reverse: "reverse (reverse xs) = xs"
  by (induct xs) (simp_all add: reverse_conc)

text {*
  The subsequent combinators make some function operate on a sequence
  in certain standard ways.
*}

fun lift :: "('a \<Rightarrow> 'b) \<Rightarrow> 'a seq \<Rightarrow> 'b seq"
where
  "lift f Empty = Empty"
| "lift f (Seq x xs) = Seq (f x) (lift f xs)"

fun iterate :: "('a \<Rightarrow> 'b \<Rightarrow> 'b) \<Rightarrow> 'a seq \<Rightarrow> 'b \<Rightarrow> 'b"
where
  "iterate f Empty a = a"
| "iterate f (Seq x xs) a = iterate f xs (f x a)"

fun iterate_reverse :: "('a \<Rightarrow> 'b \<Rightarrow> 'b) \<Rightarrow> 'a seq \<Rightarrow> 'b \<Rightarrow> 'b"
where
  "iterate_reverse f Empty a = a"
| "iterate_reverse f (Seq x xs) a = f x (iterate_reverse f xs a)"


subsection {* Exercise *}

text {*
  \<triangleright> Prove the following theorems.

  \<triangleright> Discuss issues concerning finding proofs vs. recording proofs.
*}

theorem "iterate f (conc xs ys) a = iterate f ys (iterate f xs a)" sorry
theorem "iterate f (lift g xs) = iterate (f o g) xs" sorry
theorem "iterate_reverse f xs a = iterate f (reverse xs) a" sorry


subsection {* Exercise *}

text {*
  \<triangleright> Explain what @{term "iterate \<circ> iterate"} means operationally.

  \<triangleright> Formalize this observation by suitable definition/theorem/proof.
*}

end
