header {* Compilation expressions with side-effects *}

theory Compilation
imports Main
begin

text {*
  This exercise extends the compiler example in Section~3.3 of the
  Isabelle/HOL tutorial: expressions may have side effects; see also
  6.3 of \url{http://isabelle.in.tum.de/exercises/}.
*}

subsection {* Expressions *}

text {*
  \<triangleright> Complete the subsequent definitions of expressions and
    evaluation within an environment of variable assignments.
*}

type_synonym 'v binop = "'v \<Rightarrow> 'v \<Rightarrow> 'v"

datatype ('a, 'v) exp =
    Const 'v
  | Var 'a
  | Binop "'v binop"  "('a, 'v) exp"  "('a, 'v) exp"
  | Assign 'a "('a, 'v) exp"

fun val :: "('a, 'v) exp \<Rightarrow> ('a \<Rightarrow> 'v) \<Rightarrow> 'v \<times> ('a \<Rightarrow> 'v)"
where
  "val (Const c) env = (c, env)"
| "val (Var x) env = (env x, env)"
| "val (Binop f e1 e2) env =
     (let
        (x, env1) = val e1 env;
        (y, env2) = val e2 env1
      in (f x y, env2))"
| "val (Assign a e) env =
     (let (x, env') = val e env
      in (x, env' (a := x)))"

text {*
  Pure expressions are exactly those without syntactical occurrence of
  assignment.
*}

fun pure :: "('a, 'v) exp \<Rightarrow> bool"
where
  "pure (Const c) = True"
| "pure (Var x) = True"
| "pure (Binop f e1 e2) = (pure e1 \<and> pure e2)"
| "pure (Assign x e) = False"

text {*
  \<triangleright> Produce a meaningful structured proof that evaluation of
    pure expressions does not change the environment.
    (Technically, this is a trivial induction.)
*}

theorem "pure e \<Longrightarrow> snd (val e env) = env"
proof (induct e)
  case Const
  then show ?case by simp
next
  case Var
  then show ?case by simp
next
  case (Binop f e1 e2)
  from `pure (Binop f e1 e2)` have "pure e1" and "pure e2" by simp_all
  with Binop.hyps have "snd (val e1 env) = env" and "snd (val e2 env) = env"
    by blast+
  then show ?case by (simp add: Let_def split_def)
next
  case (Assign a e)
  from `pure (Assign a e)` have False by simp
  then show ?case ..
qed


subsection {* Machine instructions *}

text {*
  \<triangleright> Regard the subsequent definitions of machine instructions
    and execution of instructions in an environment.
*}

datatype ('a, 'v) instr =
    CLoad 'v
  | VLoad 'a
  | Store 'a
  | Apply "'v binop"

fun exec :: "('a, 'v) instr list \<Rightarrow> 'v list \<Rightarrow> ('a \<Rightarrow> 'v) \<Rightarrow> 'v list \<times> ('a \<Rightarrow> 'v)"
where
  "exec [] vs hp = (vs, hp)"
| "exec (i # is) vs hp =
    (case i of
      CLoad v \<Rightarrow> exec is (v # vs) hp
    | VLoad a \<Rightarrow> exec is (hp a # vs) hp
    | Store a \<Rightarrow> exec is vs (hp (a:= hd vs))
    | Apply f \<Rightarrow> exec is (f (hd (tl vs)) (hd vs) # tl (tl vs)) hp)"

lemma
  "exec [CLoad (3::nat),
         VLoad x,
         CLoad 4,
         Apply (op *),
         Apply (op +)]
        [] (\<lambda>x. 0) = ([3], \<lambda>x. 0)"
  by simp


subsection {* Compilation *}

text {*
  \<triangleright> Complete the definition of compilation of expressions.
    Produce a structured proof for the main correctness statement.
*}

fun compile :: "('a, 'v) exp \<Rightarrow> ('a, 'v) instr list"
where
  "compile (Const c) = [CLoad c]"
| "compile (Var x) = [VLoad x]"
| "compile (Assign x e) = compile e @ [Store x]"
| "compile (Binop f e1 e2) = compile e1 @ compile e2 @ [Apply f]"

lemma exec_append:
  "exec (xs @ ys) hp vs =
    (let (hp', vs') = exec xs hp vs
     in exec ys hp' vs')"
proof (induct xs arbitrary: hp vs)
  case Nil
  show ?case
    by simp
next
  case (Cons x xs)
  then show ?case
    by (simp add: Let_def split: instr.split)
qed

lemma exec_compile:
  "exec (compile e) vs hp =
    ([fst (val e hp)] @ vs, snd (val e hp))"
proof (induct e arbitrary: vs hp)
  case (Const v)
  then show ?case by simp
next
  case (Var a)
  then show ?case by simp
next
  case (Binop f e1 e2)
  then show ?case by (simp add: exec_append Let_def split_def)
next
  case (Assign a e)
  then show ?case
    by (simp add: exec_append Let_def split_def) (simp add: fun_upd_def)
qed

theorem correctness:
    "exec (compile e) [] s = ([fst (val e s)], snd (val e s))"
  by (simp add: exec_compile)

end
