
(* Fichier principal de l'interprte mini-Turtle *)

open Format
open Lexing

(* Option de compilation, pour s'arrter  l'issue du parser *)
let parse_only = ref false

(* Noms des fichiers source et cible *)
let ifile = ref ""
let ofile = ref ""

let set_file f s = f := s

(* Les options du compilateur que l'on affiche avec --help *)
let options =
  ["--parse-only", Arg.Set parse_only,
   "  Pour ne faire uniquement que la phase d'analyse syntaxique"]

let usage = "usage: mini-turtle [option] file.logo"

(* localise une erreur en indiquant la ligne et la colonne *)
let localisation pos =
  let l = pos.pos_lnum in
  let c = pos.pos_cnum - pos.pos_bol + 1 in
  eprintf "File \"%s\", line %d, characters %d-%d:\n" !ifile l (c-1) c

let () =
  (* Parsing de la ligne de commande *)
  Arg.parse options (set_file ifile) usage;

  (* On vrifie que le nom du fichier source a bien t indiqu *)
  if !ifile="" then begin eprintf "Aucun fichier  compiler\n@?"; exit 1 end;

  (* Ce fichier doit avoir l'extension .logo *)
  if not (Filename.check_suffix !ifile ".logo") then begin
    eprintf "Le fichier d'entre doit avoir l'extension .logo\n@?";
    Arg.usage options usage;
    exit 1
  end;

  (* Ouverture du fichier source en lecture *)
  let f = open_in !ifile in

  (* Cration d'un tampon d'analyse lexicale *)
  let buf = Lexing.from_channel f in

  try
    (* Parsing: la fonction  Parser.prog transforme le tampon lexical en un
       arbre de syntaxe abstraite si aucune erreur (lexicale ou syntaxique)
       n'est dtecte.
       La fonction Lexer.token est utilise par Parser.prog pour obtenir
       le prochain token. *)
    let p = Parser.prog Lexer.token buf in
    close_in f;

    (* On s'arrte ici si on ne veut faire que le parsing *)
    if !parse_only then exit 0;

    Interp.prog p
  with
    | Lexer.Lexing_error c ->
	(* Erreur lexicale. On rcupre sa position absolue et
	   on la convertit en numro de ligne *)
	localisation (Lexing.lexeme_start_p buf);
	eprintf "Erreur lexicale: %s@." c;
	exit 1
    | Parser.Error ->
	(* Erreur syntaxique. On rcupre sa position absolue et on la
	   convertit en numro de ligne *)
	localisation (Lexing.lexeme_start_p buf);
	eprintf "Erreur syntaxique@.";
	exit 1
    | Interp.Error s->
	(* Erreur pendant l'interprtation *)
	eprintf "Erreur : %s@." s;
	exit 1
